import numpy as np
import matplotlib.pyplot as plt
from numpy.linalg import svd, norm
#We perform Monte Carlo experiments on a synthetic PSD matrix with exponentially decaying spectrum
# A = synthetic PSD matrix with exponentially decaying spectrum with n =50
# E = Gaussian Noise
#p = 5
# We evaluate (i) Spectral-norm error (ii) Frobenius-norm error (iii) Change-in-error

#Generate A
np.random.seed(42)

# Parameters
n, p = 50, 5
num_trials   = 20
noise_levels = np.linspace(0, 2.0, 20)

# PSD test matrix with geometric spectrum
decay = 0.8
σ = decay ** np.arange(n)
U = np.linalg.qr(np.random.randn(n, n))[0]
A   = U @ np.diag(σ) @ U.T
A_p = U[:, :p] @ np.diag(σ[:p]) @ U[:, :p].T

spec_m, fro_m, diff_m = [], [], []
spec_sd, fro_sd, diff_sd = [], [], []

for σ_noise in noise_levels:
    s_trial, f_trial, d_trial = [], [], []

    for _ in range(num_trials):
        #Geneate Gaussian noise
        E = np.random.normal(0, σ_noise, size=(n, n))
        E = 0.5 * (E + E.T)
        A_tilde = A + E
# Compute (i) Spectral-norm error (ii) Frobenius-norm error (iii) Change-in-error
        U_t, S_t, _ = svd(A_tilde, full_matrices=False)
        A_tilde_p = U_t[:, :p] @ np.diag(S_t[:p]) @ U_t[:, :p].T

        s_trial.append(norm(A_tilde_p - A_p, 2)) #Spectral norm
        f_trial.append(norm(A_tilde_p - A_p, 'fro')) #Fro norm
        d_trial.append(abs(norm(A - A_p, 2) - norm(A - A_tilde_p, 2))) #change-in-error

    spec_m.append(np.mean(s_trial));  spec_sd.append(np.std(s_trial, ddof=1))
    fro_m .append(np.mean(f_trial));  fro_sd .append(np.std(f_trial, ddof=1))
    diff_m.append(np.mean(d_trial)); diff_sd.append(np.std(d_trial, ddof=1))

# --- plot with capsized error bars ---------------------------------
plt.figure(figsize=(6, 5))
plt.errorbar(noise_levels, spec_m,  yerr=spec_sd,  fmt='o-', lw=2, ms=5,
             capsize=3, label=r'$\|\tilde{A}_p - A_p\|_2$')
plt.errorbar(noise_levels, fro_m,   yerr=fro_sd,   fmt='s--', lw=2, ms=5,
             capsize=3, label=r'$\|\tilde{A}_p - A_p\|_F$')
plt.errorbar(noise_levels, diff_m,  yerr=diff_sd,  fmt='d-.', lw=2, ms=5,
             capsize=3, label=r'$|\|A-A_p\| - \|A-\tilde{A}_p\||$')

plt.xlabel("Noise level $\sigma$")
plt.ylabel("Error")

plt.grid(True)
plt.tight_layout()
plt.show()